#region References

using System;
using System.Collections;
using System.Data;
using System.Linq;
using System.Text;

using Method = System.Reflection.MethodBase;

using gov.va.med.vbecs;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;

using BLOODUNITFINANCE = gov.va.med.vbecs.Common.VbecsTables.BloodUnitFinance;
using BLOODUNITSTATUS = gov.va.med.vbecs.Common.VbecsTables.BloodUnitStatus;

#endregion

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carl Jensen</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/30/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Summary description for DiscardOrQuarantineBloodUnit.</summary>

	#endregion	

	public class DiscardOrQuarantineBloodUnit : BaseBusinessObject
	{
		#region Variables

		private const string TEST_TYPE_ABO = "ABO";
		private const string TEST_TYPE_RH = "Rh";

		///	<summary>
		///	Collection of BloodUnits being operated on.
		///	</summary>
		private ArrayList _bloodUnitArray;
		///	<summary>
		///	Collection of BloodUnits that are valid for submission.
		///	</summary>
		private ArrayList _validUnits;
		///	<summary>
		///	Free text comments.
		///	</summary>
		private string _details = String.Empty;
		///	<summary>
		///	Detail code when discard chosen.
		///	</summary>
		private Common.DiscardDisposition  _discardDisposition = Common.DiscardDisposition.Waste;
		///	<summary>
		///	Canned comment text
		///	</summary>
		private string _commentText = string.Empty;
		///	<summary>
		///	Canned comment guid
		///	</summary>
		private Guid _commentGuid = Guid.Empty;
		///	<summary>
		///	Holds user choice; discard, quarantine or remove from quarantine.
		///	</summary>
		private Common.Action _action = Common.Action.Discard;
		///	<summary>
		///	Holds the date/time user has selected.
		///	</summary>
		private System.DateTime _userDateTime = VBECSDateTime.GetDivisionCurrentDateTime();

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="744"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>None/New object must be created.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty DiscardOrQuarantineBloodUnit constructor
		/// </summary>
		public DiscardOrQuarantineBloodUnit()
		{
			this._bloodUnitArray = new ArrayList();
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="876"> 
		///		<ExpectedInput>Valid comment</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2175"> 
		///		<ExpectedInput>string.Empty</ExpectedInput>
		///		<ExpectedOutput>Broken rule</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2176"> 
		///		<ExpectedInput>CommentText = Other and empty details</ExpectedInput>
		///		<ExpectedOutput>Different rule broken</ExpectedOutput>
		///</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Discard/Quarantine/Release canned comment text
		/// </summary>
		public string CommentText
		{
			get
			{
				return _commentText;
			}
			set
			{
				_commentText = value;
				RuleBroken(Method.GetCurrentMethod().Name, this._commentText.Trim().Equals(string.Empty));
				RuleBroken("Details", this._commentText.ToUpper().Trim().Equals("OTHER")&& this._details.Equals(String.Empty));
			}
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="877"> 
		///		<ExpectedInput>Valid comment</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2178"> 
		///		<ExpectedInput>Guid.Empty</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Discard/Quarantine/Release canned comment guid
		/// </summary>
		public Guid CommentGuid
		{
			get
			{
				return _commentGuid;
			}
			set
			{
				_commentGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="878"> 
		///		<ExpectedInput>Detail is not empty</ExpectedInput>
		///		<ExpectedOutput>No  broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2186"> 
		///		<ExpectedInput>CommentText = Other and empty details</ExpectedInput>
		///		<ExpectedOutput>Broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Details 
		/// </summary>
		public string Details
		{
			get
			{
				return _details;
			}
			set
			{
				_details = value;
				RuleBroken("Details",(this._commentText.ToUpper().Trim().Equals("OTHER") && this._details.Trim().Length == 0));
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="879"> 
		///		<ExpectedInput>Valid action</ExpectedInput>
		///		<ExpectedOutput>Property set to a valid action</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2205"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Action
		/// </summary>
		public Common.Action Action
		{
			get
			{
				return _action;
			}
			set
			{
				_action = value;
			}	
		}

		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="880"> 
		///		<ExpectedInput>Valid discard disposition</ExpectedInput>
		///		<ExpectedOutput>Valid discard disposition</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="7"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Discard Disposition
		/// </summary>
		public Common.DiscardDisposition DiscardDisposition
		{
			get
			{
				return _discardDisposition;
			}
			set
			{
				_discardDisposition = value;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="881"> 
		///		<ExpectedInput>ArrayList</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="4"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Blood Units
		/// </summary>
		public ArrayList BloodUnits
		{
			get 
			{ 
				return _bloodUnitArray;
			}
			set 
			{
				_bloodUnitArray = value;
			}
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="882"> 
		///		<ExpectedInput>Valid DateTime</ExpectedInput>
		///		<ExpectedOutput>Valid Datetime</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="12"> 
		///		<ExpectedInput>Future DateTime</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// User Date/Time 
		/// </summary>
		public System.DateTime UserDateTime
		{
			get 
			{ 
				return _userDateTime;
			}
			set 
			{
				_userDateTime = value;
				RuleBroken(Method.GetCurrentMethod().Name, value > VBECSDateTime.GetDivisionCurrentDateTime());
			}
		}
		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="975"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing Quarantine Details</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="8"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves comments with QR (release from quarantine) 
		/// or QD (quarantine or discard) context
		/// BR_27.04, BR_27.08
		/// </summary>
		/// <param name="context">QD (quarantine or discard) or QR (release from quarantine)</param>
		/// <returns>DataTable</returns>
		public static DataTable GetComboBoxEntries(string context) 
		{
			return DAL.CannedComment.GetCannedComments(Common.LogonUser.LogonUserDivisionCode, context);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="976"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing Statuses</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="10"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Fills status combo box
		/// </summary>
		/// <returns></returns>
		///
		public static DataTable PopulateStatusComboBox()
		{
			DataTable dtStatusDetails = DAL.DiscardOrQuarantineBloodUnit.GetStatusDetails();
			return dtStatusDetails;
		}

		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="978"> 
		///		<ExpectedInput>Valid BloodUnit object</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2167"> 
		///		<ExpectedInput>null bloodUnit</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads a blood Unit object from a Discard or Quarantine unit DataRow
		/// </summary>
		/// <param name="bloodUnit"></param>
		public void LoadBloodUnitFields(BOL.BloodUnit bloodUnit)
		{
			if (bloodUnit == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("Blood Unit").ResString);
			}
			//
			DataRow drDiscardQuarantineData;
			drDiscardQuarantineData = DAL.DiscardOrQuarantineBloodUnit.GetDiscardOrQuarantineData(bloodUnit.BloodUnitGuid);
			//
			bloodUnit.BloodUnitFinance.ReturnCreditAmount = drDiscardQuarantineData.IsNull(Common.VbecsTables.BloodUnitFinance.ReturnCreditAmount) ? 0 : (decimal) drDiscardQuarantineData[Common.VbecsTables.BloodUnitFinance.ReturnCreditAmount];
			bloodUnit.BloodUnitMedia.UnitExpirationDate = (System.DateTime) drDiscardQuarantineData[Common.VbecsTables.BloodUnitMedia.UnitExpirationDate];
			bloodUnit.DirectedPatient.PatientGuid = drDiscardQuarantineData.IsNull(Common.VbecsTables.BloodUnitMedia.DirectedPatientGuid) ? System.Guid.Empty : (System.Guid) drDiscardQuarantineData[Common.VbecsTables.BloodUnitMedia.DirectedPatientGuid];
			bloodUnit.DirectedPatient.FirstName = drDiscardQuarantineData.IsNull(Common.VbecsTables.Patient.PatientFirstName) ? String.Empty : (string) drDiscardQuarantineData[Common.VbecsTables.Patient.PatientFirstName];
			bloodUnit.DirectedPatient.LastName = drDiscardQuarantineData.IsNull(Common.VbecsTables.Patient.PatientLastName) ? String.Empty : (string) drDiscardQuarantineData[Common.VbecsTables.Patient.PatientLastName];
			bloodUnit.BloodUnitStatus.QuarantineIndicator = drDiscardQuarantineData.IsNull(BLOODUNITSTATUS.QuarantineIndicator) ? false : (bool) drDiscardQuarantineData[BLOODUNITSTATUS.QuarantineIndicator];
			bloodUnit.ProductType.ProductTypeCode = drDiscardQuarantineData.IsNull(Common.VbecsTables.BloodProduct.ProductTypeCode) ? String.Empty : (string) drDiscardQuarantineData[Common.VbecsTables.BloodProduct.ProductTypeCode];
			bloodUnit.LastUpdateDate = drDiscardQuarantineData.IsNull(Common.VbecsTables.BloodUnit.LastUpdateDate) ? DateTime.MinValue : (DateTime) drDiscardQuarantineData[Common.VbecsTables.BloodUnit.LastUpdateDate];
			//
			if (!drDiscardQuarantineData.IsNull(Common.DatabaseConstants.ArtificialColumnNames.UnitStatusRowVersion))
			{
				bloodUnit.BloodUnitStatus.RowVersion = (byte[]) drDiscardQuarantineData[Common.DatabaseConstants.ArtificialColumnNames.UnitStatusRowVersion];
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="883"> 
		///		<ExpectedInput>Arraylist, bool, UpdateFunction, WorkloadProcessId</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Method that updates the quarantine status in BloodUnit
		/// Implements BR_27.15
		/// </summary>
		/// <param name="quarantineExceptions"></param>
		/// <param name="quarantine"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="workloadProcessIDs">process to generate workload for</param>
		public bool ChangeQuarantineStatus(ArrayList quarantineExceptions, bool quarantine, Common.UpdateFunction lastUpdateFunctionId, ArrayList workloadProcessIDs)
		{
			int assignedUnitCount = 0;
			System.Data.DataTable dtWorkloadEvents = null;
			System.Data.DataTable dtWorkloadTemp = null;
			DataTable dtQuarantineExceptions = null;
			DataTable dtQuarantine = new DataTable(BLOODUNITSTATUS.TableName);
			DataTable dtOrderedUnit = null;
			// CR 2941
			ArrayList releasedUnitsForBCE = new ArrayList();

			dtQuarantine.Columns.Add(BLOODUNITSTATUS.BloodUnitGuid, typeof(Guid));
			dtQuarantine.Columns.Add(Common.VbecsTables.VamcDivision.DivisionCode, typeof(string));
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.LastUpdateFunctionId, typeof(int));
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.LastUpdateUser);
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.QuarantineCannedCommentGuid, typeof(Guid));
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.QuarantineIndicator, typeof(bool));
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.RowVersion, typeof(byte[]));
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.UnitStatusDate, typeof(DateTime));
			dtQuarantine.Columns.Add(BLOODUNITSTATUS.UnitStatusComments);

			try
			{
				System.Guid [] bloodUnitGuids = new System.Guid [_bloodUnitArray.Count];
				int nCtr = 0;
				foreach (BOL.BloodUnit bloodUnit in this._bloodUnitArray) 
				{
					if ( bloodUnit.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Assigned
						|| bloodUnit.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.CrossmatchedAssigned )
					{
						assignedUnitCount += 1;

						// CR 2941
						DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(bloodUnit.BloodUnitGuid);

						for (int j=0; j< dtPatients.Rows.Count; j++)
						{
							releasedUnitsForBCE.Add(bloodUnit.BloodUnitGuid);
							releasedUnitsForBCE.Add((Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid]);
						}
					}
					//
					DataRow drQuarantine = dtQuarantine.NewRow();
					bloodUnitGuids[nCtr] = bloodUnit.BloodUnitGuid;
					drQuarantine[BLOODUNITSTATUS.BloodUnitGuid] = bloodUnit.BloodUnitGuid;
					drQuarantine[Common.VbecsTables.VamcDivision.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drQuarantine[BLOODUNITSTATUS.LastUpdateFunctionId] = lastUpdateFunctionId;
					drQuarantine[BLOODUNITSTATUS.LastUpdateUser] = Common.LogonUser.LogonUserName;
					//
					if(this.CommentGuid.Equals(Guid.Empty))
					{
						drQuarantine[BLOODUNITSTATUS.QuarantineCannedCommentGuid] = DBNull.Value;
					}
					else
					{
						drQuarantine[BLOODUNITSTATUS.QuarantineCannedCommentGuid] = this.CommentGuid;
					}
					//
					drQuarantine[BLOODUNITSTATUS.QuarantineIndicator] = quarantine;
					drQuarantine[BLOODUNITSTATUS.RowVersion] = bloodUnit.BloodUnitStatus.RowVersion;
					drQuarantine[BLOODUNITSTATUS.UnitStatusDate] = this._userDateTime;
					drQuarantine[BLOODUNITSTATUS.UnitStatusComments] = this._details;

					dtQuarantine.Rows.Add(drQuarantine);
					nCtr++;
				}

				foreach(BOL.ExceptionReport exRep in quarantineExceptions)
				{
					if (dtQuarantineExceptions == null)
					{
						dtQuarantineExceptions = exRep.ExceptionData.Table.Clone();
					}
					dtQuarantineExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
				}

				// BR_27.15:	Unit has had a quarantine indicator assigned, 
				//				the system will release any patient assignment.
				//				Any patient restriction associated with the  
				//				unit remains in tact.
				if ( assignedUnitCount > 0 )
				{
					dtOrderedUnit = OrderedUnit.GetOrderedUnitDataTableForReleaseOfPatientAssignment( this._bloodUnitArray, BOL.VBECSDateTime.GetDivisionCurrentDateTime(), LogonUser.LogonUserName );
					dtOrderedUnit = Common.Utility.AppendLastUpdateInformation( dtOrderedUnit, Common.UpdateFunction.UC027FrmDiscardQuarantine );
				}
				

				for (int i=0; i<workloadProcessIDs.Count; i++)
				{
					if (dtWorkloadEvents == null)
					{
						dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIDs[i], bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
					}
					else
					{
						dtWorkloadTemp = BOL.WorkloadEvent.GenerateWorkloadData((Common.WorkloadProcessID)workloadProcessIDs[i], bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
						if (dtWorkloadTemp.Rows.Count > 0)
						{
							dtWorkloadEvents.Rows.Add(dtWorkloadTemp.Rows[0].ItemArray);
						}
					}
				}

				Common.Utility.AppendLastUpdateInformation( dtWorkloadEvents, lastUpdateFunctionId );

				DataTable dtBCEMessages = null;
				DataTable dtTempBCEMessages = null;

				//CR 2941
				//CR 2945
				//generate table with BCE_MESSAGE
				if (releasedUnitsForBCE.Count > 0)
				{
					for (int i=0; i<releasedUnitsForBCE.Count; i=i+2)
					{
						//CR 3015
						if (dtBCEMessages == null)
						{
							dtBCEMessages = BOL.OrderedUnit.UpdateBceForRelease((Guid)releasedUnitsForBCE[i], (Guid)releasedUnitsForBCE[i+1], false);
						}
						else
						{
							dtTempBCEMessages = BOL.OrderedUnit.UpdateBceForRelease((Guid)releasedUnitsForBCE[i], (Guid)releasedUnitsForBCE[i+1], false);
							if (dtTempBCEMessages.Rows.Count > 0)
								dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
						}
					}
				}
				//BCE_MESSAGE end

				bool result =  DAL.DiscardOrQuarantineBloodUnit.UpdateQuarantineStatus( dtQuarantine, dtWorkloadEvents, dtQuarantineExceptions, dtOrderedUnit, dtBCEMessages );
				
				return result;
			}
			catch(Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new BusinessObjectException(string.Concat(Method.GetCurrentMethod().Name,": ", ex.Message),ex);
			}
		}
		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5"> 
		///		<ExpectedInput>valid exception, credit, UpdateFunction, WorkloadProcessId</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="884"> 
		///		<ExpectedInput>valid exception, no credit, UpdateFunction, WorkloadProcessId</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Changes discard status of a unit 
		/// </summary>
		/// <param name="discardExceptions"></param>
		/// <param name="credit"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="workloadProcessId"></param>
		public bool ChangeDiscardStatus(ArrayList discardExceptions, bool credit, Common.UpdateFunction lastUpdateFunctionId, Common.WorkloadProcessID workloadProcessId)
		{
			DataTable dtDiscardExceptions = null;
			//
			DataTable dtFinance = new DataTable(BLOODUNITFINANCE.TableName);
			dtFinance.Columns.Add(BLOODUNITFINANCE.BloodUnitFinanceGuid, typeof(Guid));
			dtFinance.Columns.Add(BLOODUNITFINANCE.ReturnCreditAmount, typeof(decimal));
			dtFinance.Columns.Add(BLOODUNITFINANCE.LastUpdateUser);
			dtFinance.Columns.Add(BLOODUNITFINANCE.LastUpdateFunctionId, typeof(int));
			dtFinance.Columns.Add(BLOODUNITFINANCE.RowVersion, typeof(byte[]));
			//
			DataTable dtDiscard = new DataTable(BLOODUNITSTATUS.TableName);
			dtDiscard.Columns.Add(BLOODUNITSTATUS.BloodUnitStatusGuid, typeof(Guid));
			dtDiscard.Columns.Add(BLOODUNITSTATUS.UnitStatusDate, typeof(DateTime));
			dtDiscard.Columns.Add(BLOODUNITSTATUS.UnitStatusComments);
			dtDiscard.Columns.Add(BLOODUNITSTATUS.DiscardCannedCommentGuid, typeof(Guid));
			dtDiscard.Columns.Add(BLOODUNITSTATUS.LastUpdateUser);
			dtDiscard.Columns.Add(BLOODUNITSTATUS.LastUpdateFunctionId, typeof(int));
			dtDiscard.Columns.Add(BLOODUNITSTATUS.RowVersion, typeof(byte[]));
			dtDiscard.Columns.Add(Common.VbecsTables.VamcDivision.DivisionCode, typeof(string));
			//
			try
			{
				System.Guid [] bloodUnitGuids = new System.Guid [_bloodUnitArray.Count];
				int nCtr = 0;
				foreach (BOL.BloodUnit bloodUnit in this._bloodUnitArray) 
				{
					Decimal returnCredit = 0;
					if (credit) 
					{
						returnCredit = bloodUnit.BloodUnitFinance.ReturnCreditAmount;
					}
					//
					DataRow drFinance = dtFinance.NewRow();
					drFinance[BLOODUNITFINANCE.BloodUnitFinanceGuid] = bloodUnit.BloodUnitFinance.BloodUnitFinanceGuid;
					drFinance[BLOODUNITFINANCE.ReturnCreditAmount] = returnCredit;
					drFinance[BLOODUNITFINANCE.LastUpdateUser] = Common.LogonUser.LogonUserName;
					drFinance[BLOODUNITFINANCE.LastUpdateFunctionId] = lastUpdateFunctionId;
					drFinance[BLOODUNITFINANCE.RowVersion] = bloodUnit.BloodUnitFinance.RowVersion;
					dtFinance.Rows.Add(drFinance);
					//
					DataRow drDiscard = dtDiscard.NewRow();
					bloodUnitGuids[nCtr] = bloodUnit.BloodUnitGuid;
					drDiscard[BLOODUNITSTATUS.BloodUnitStatusGuid] = bloodUnit.BloodUnitStatus.BloodUnitStatusGuid;
					drDiscard[BLOODUNITSTATUS.UnitStatusDate] = this._userDateTime;
					drDiscard[BLOODUNITSTATUS.UnitStatusComments] = this._details;
					drDiscard[BLOODUNITSTATUS.DiscardCannedCommentGuid] = this.CommentGuid;
					drDiscard[BLOODUNITSTATUS.LastUpdateUser] = Common.LogonUser.LogonUserName;
					drDiscard[BLOODUNITSTATUS.LastUpdateFunctionId] = lastUpdateFunctionId;
					drDiscard[Common.VbecsTables.VamcDivision.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					drDiscard[BLOODUNITSTATUS.RowVersion] = bloodUnit.BloodUnitStatus.RowVersion;
					dtDiscard.Rows.Add(drDiscard);
					//
					nCtr++;
				}

				foreach(BOL.ExceptionReport exRep in discardExceptions)
				{
					if (dtDiscardExceptions == null)
					{
						dtDiscardExceptions = exRep.ExceptionData.Table.Clone();
					}
					dtDiscardExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
				}

				DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
				Common.Utility.AppendLastUpdateInformation(dtWorkload, lastUpdateFunctionId);
				DataTable dtBCEMessages = null;
				DataTable dtTempBCEMessages = null;

				//CR 2941
				//CR 2945
				//generate table with BCE_MESSAGE
				//The Unit is removed from assignment when it is discarded out

				Guid bloodUnitGuid = Guid.Empty;
				Guid patientGuid = Guid.Empty;
				for (int i=0; i< _bloodUnitArray.Count; i++)
				{
					bloodUnitGuid = ((BOL.BloodUnit)_bloodUnitArray[i]).BloodUnitGuid;

					//getting all patients assigned to a unit
					DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(bloodUnitGuid);

					for (int j=0; j< dtPatients.Rows.Count; j++)
					{
						patientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];
						
						//CR 3015
						if (dtBCEMessages == null)
						{
							dtBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnitGuid, patientGuid, false);
						}
						else
						{
							dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnitGuid, patientGuid, false);
							if (dtTempBCEMessages.Rows.Count > 0)
								dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
						}
					}
				}
				//BCE_MESSAGE end

				bool result =  DAL.DiscardOrQuarantineBloodUnit.UpdateDiscardStatus(dtFinance, dtDiscard, dtDiscardExceptions, dtWorkload, dtBCEMessages);

				return result;
			}
			catch(Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new BusinessObjectException(string.Concat(Method.GetCurrentMethod().Name,": ", ex.Message),ex);
			}
		}

		///<Developers>
		///<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="979"> 
		///		<ExpectedInput>Valid bloodUnit for discard</ExpectedInput>
		///		<ExpectedOutput>Unit is determined to be valid for discard</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="2371"> 
		///		<ExpectedInput>Valid bloodUnit for quarantine</ExpectedInput>
		///		<ExpectedOutput>Unit is determined to be valid for quarantine</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="2372"> 
		///		<ExpectedInput>Valid bloodUnit for removal from quarantine</ExpectedInput>
		///		<ExpectedOutput>Unit is determined to be valid for removal from quarantine</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2373"> 
		///		<ExpectedInput>Invalid bloodUnit for discard</ExpectedInput>
		///		<ExpectedOutput>Unit is determined to be invalid for discard</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2374"> 
		///		<ExpectedInput>Invalid bloodUnit for quarantine</ExpectedInput>
		///		<ExpectedOutput>Unit is determined to be invalid for quarantine</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2375"> 
		///		<ExpectedInput>Invalid bloodUnit for removal from quarantine</ExpectedInput>
		///		<ExpectedOutput>Unit is determined to be invalid for removal from quarantine</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Validates a batch submission 
		/// Implements BR_27.03, BR_27.16
		/// </summary>
		/// <param name="action"></param>
		/// <param name="failureMessages"></param>
		/// <param name="warningMessages"></param>
		/// <param name="expiredUnits"></param>
		/// <param name="crossMatchedUnits"></param>
		/// <param name="expired"></param>
		/// <param name="userDate"></param>
		/// <returns></returns>
		public ArrayList ValidateBatchSubmission(Common.Action action, StringBuilder failureMessages, StringBuilder warningMessages, ArrayList expiredUnits, ArrayList crossMatchedUnits, StringBuilder expired, ref DateTime[] userDate)
		{
			// holds warnings
			StringBuilder warningMessagesTemp = new StringBuilder();

			// holds reasons for failure
			StringBuilder failureMessagesTemp = new StringBuilder();

			// holds units that pass rules for submission
			_validUnits = new ArrayList();

			int originalUnitCount = this._bloodUnitArray.Count;

			// set up failure message depending on the action
			if (this._action == Common.Action.Quarantine) failureMessagesTemp.Append("The following unit(s) could not be quarantined: \n");
			if (this._action == Common.Action.Discard) failureMessagesTemp.Append("The following unit(s) could not be discarded: \n");
			if (this._action == Common.Action.RemoveFromQuarantine) failureMessagesTemp.Append("The following unit(s) could not be removed from quarantine: \n");

            DateTime dt = System.DateTime.Now;
			// Iterate through blood Units and determine if they are eligible for submission.  
			// If not, generate error messages.
			BloodUnit [] bloodUnits = (BloodUnit [])this._bloodUnitArray.ToArray(typeof(BloodUnit));
			for (int idx = 0; idx < bloodUnits.Length && idx < userDate.Length; idx++)
			{
                //CR 3554 DateTime is a Value type so it can never be null but to try to satisfy fortify we check things, fortify's bug is still flagging this as null dereference
                if (userDate != null && idx < userDate.Length && userDate[idx] != null) dt = userDate[idx];
                
				//CR 3195 modified this code to return the date/time of changed record
				if ( DateTime.Compare(bloodUnits[idx].LastUpdateDate, _userDateTime ) > 0)
				{
                    if (bloodUnits[idx].LastUpdateDate > dt)
                        if (userDate != null) userDate[idx] = bloodUnits[idx].LastUpdateDate;
				}
				if ( DateTime.Compare(bloodUnits[idx].BloodUnitStatus.LastUpdateDate, _userDateTime) > 0)
				{
                    if (bloodUnits[idx].BloodUnitStatus.LastUpdateDate > dt)
                        if (userDate != null) userDate[idx] = bloodUnits[idx].BloodUnitStatus.LastUpdateDate;
				}
				if ( DateTime.Compare(bloodUnits[idx].BloodUnitStatus.CurrentStatusLastUpdateDate, _userDateTime) > 0)
				{
                    if (bloodUnits[idx].BloodUnitStatus.CurrentStatusLastUpdateDate > dt)
                        if (userDate != null) userDate[idx] = bloodUnits[idx].BloodUnitStatus.CurrentStatusLastUpdateDate;
				}
				if ( DateTime.Compare(bloodUnits[idx].BloodUnitMedia.LastUpdateDate, _userDateTime) > 0)
				{
                    if (bloodUnits[idx].BloodUnitMedia.LastUpdateDate > dt)
                        if (userDate != null) userDate[idx] = bloodUnits[idx].BloodUnitMedia.LastUpdateDate;
				}
				if ( DateTime.Compare(bloodUnits[idx].BloodUnitFinance.LastUpdateDate, _userDateTime) > 0)
				{
					if (bloodUnits[idx].BloodUnitFinance.LastUpdateDate > dt)
                        if (userDate != null) userDate[idx] = bloodUnits[idx].BloodUnitFinance.LastUpdateDate;
				}
					//checking BloodUnitTest records
				
				DataTable dtUnitTests = BOL.BloodUnitTest.GetBloodUnitTests(bloodUnits[idx].BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
				if (dtUnitTests.Rows.Count > 0)
				{
					//getting the most recently changed BloodUnitTest record
					DataRow[] drUnitTests = dtUnitTests.Select("", Common.VbecsTables.BloodUnitTest.TestDate + " DESC");
					if (!drUnitTests[0].IsNull(Common.VbecsTables.BloodUnitTest.TestDate) &&
						DateTime.Compare((DateTime)drUnitTests[0][Common.VbecsTables.BloodUnitTest.TestDate], _userDateTime) > 0)
					{
                        if ((DateTime)drUnitTests[0][Common.VbecsTables.BloodUnitTest.TestDate] > dt)
                            if (userDate != null) userDate[idx] = (DateTime)drUnitTests[0][Common.VbecsTables.BloodUnitTest.TestDate];
					}
				}
				//
				_validUnits.Add(bloodUnits[idx]);
				Common.UnitStatusCode unitStatusCode = bloodUnits[idx].BloodUnitStatus.UnitStatusCode;

				string status = string.Empty;
				// Quarantine
				if (this._action == Common.Action.Quarantine)
				{
					status = "quarantine";
					if ( bloodUnits[idx].BloodUnitStatus.QuarantineIndicator == true )
					{
						failureMessagesTemp.Append(Common.StrRes.SysErrMsg.UC027.UnitAlreadyQuarantined(bloodUnits[idx].EyeReadableUnitId).ResString);
						this._validUnits.Remove(bloodUnits[idx]);
					}
				}
				//
				if (this._action == Common.Action.Discard)
				{
					status = "discard";
				}
				// note: the system DOES allow discards of assigned/crossmatched units
				if ( this._action == Common.Action.Quarantine 
					&& ( unitStatusCode == Common.UnitStatusCode.Assigned 
						|| unitStatusCode == Common.UnitStatusCode.CrossmatchedAssigned) )
				{
					warningMessagesTemp.Append( "The unit has been assigned/crossmatched, so if you quarantine it, then we will need to remove the assignment.\n");
				}
				//
				if (this._action == Common.Action.Quarantine || this._action == Common.Action.Discard)
				{
					if ( unitStatusCode == Common.UnitStatusCode.Transferred
						|| unitStatusCode == Common.UnitStatusCode.Transfused
						|| unitStatusCode == Common.UnitStatusCode.Discarded)
					{
						failureMessagesTemp.Append(Common.StrRes.SysErrMsg.UC027.UnitInFinalStatus(status, bloodUnits[idx].EyeReadableUnitId).ResString + "\n");
						this._validUnits.Remove(bloodUnits[idx]);
					}
				}
				// remove from quarantine
				if (this._action == Common.Action.RemoveFromQuarantine)
				{
					bool valid = true;
					//
					// BR_27.03
					if (!bloodUnits[idx].BloodUnitStatus.QuarantineIndicator)
					{
						failureMessagesTemp.Append(Common.StrRes.SysErrMsg.UC027.UnitNotQuarantined(bloodUnits[idx].EyeReadableUnitId).ResString);
						failureMessagesTemp.Append("\n");
						valid = false;
					}
					// BR_27.16
					if(this.IsUnitAboRhInconsistent(bloodUnits[idx].BloodUnitGuid))
					{
						failureMessagesTemp.Append(Common.StrRes.SysErrMsg.UC027.InconsistentUnitAboRh(bloodUnits[idx].EyeReadableUnitId).ResString);
						failureMessagesTemp.Append("\n");
						valid = false;
					}
					//
					// BR_27.16
					if(this.IsUnitAntigenTypingInconsistent(bloodUnits[idx].BloodUnitGuid))
					{
						failureMessagesTemp.Append(Common.StrRes.SysErrMsg.UC027.InconsistentUnitAntigenTyping(bloodUnits[idx].EyeReadableUnitId).ResString);
						failureMessagesTemp.Append("\n");
						valid = false;
					}
					//
					if (!valid)
					{
						this._validUnits.Remove(bloodUnits[idx]);
					}
					else
					{
						if (bloodUnits[idx].BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Assigned || bloodUnits[idx].BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.CrossmatchedAssigned)
						{
							crossMatchedUnits.Add(bloodUnits[idx]);
						}
						// date processing
						if ( DateTime.Compare(VBECSDateTime.GetDivisionCurrentDateTime(), bloodUnits[idx].BloodUnitMedia.UnitExpirationDate ) > 0 )
						{
							expiredUnits.Add(bloodUnits[idx]);
							if ( bloodUnits[idx].BloodUnitStatus.UnitStatusCode != Common.UnitStatusCode.CrossmatchedAssigned)
							{
								expired.Append(bloodUnits[idx].EyeReadableUnitId).Append("\n");						
							} 
							else
							{
								this._validUnits.Remove(bloodUnits[idx]);
								expiredUnits.Remove(bloodUnits[idx]);
								failureMessagesTemp.Append(bloodUnits[idx].EyeReadableUnitId).Append(" - Crossmatched unit has expired.\n");
							}
						}
					} // else
				} // if remove from quarantine
			} // foreach

			// return warnings
			warningMessages.Append( warningMessagesTemp );
			//
			// if there has been a failure, add the failure message to the main message
			if (originalUnitCount > this._validUnits.Count) 
			{
				failureMessages.Append( failureMessagesTemp );
			}
			return this._validUnits;
		}

		///<Developers>
		///	<Developer>COE User</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3534"> 
		///		<ExpectedInput>Arraylist of BloodUnits</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3535"> 
		///		<ExpectedInput>Empty Arraylist</ExpectedInput>
		///		<ExpectedOutput>NoValid string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BuildValidString
		/// </summary>
		/// <param name="bloodUnits"></param>
		/// <returns></returns>
		public string BuildValidString(ArrayList bloodUnits)
		{
			if (bloodUnits.Count > 0) 
			{
				StringBuilder validString = new StringBuilder();
				switch (this._action)
				{
					case Common.Action.Quarantine:
						validString.Append(Common.StrRes.InfoMsg.Common.QuarantineUnits().ResString).Append("\n");
						break;
					case Common.Action.RemoveFromQuarantine:
						validString.Append(Common.StrRes.InfoMsg.Common.RemovefromQuarantineUnits().ResString).Append("\n");
						break;
					default:
						// Discard is default
						validString.Append(Common.StrRes.InfoMsg.Common.DiscardUnits().ResString).Append("\n");
						break;
				}
				foreach (BOL.BloodUnit bloodUnit in bloodUnits) 
				{
					validString.Append(bloodUnit.EyeReadableUnitId).Append("\n");
				}
				return validString.ToString();
			}
			else
			{
				return Common.StrRes.InfoMsg.Common.NoValidUnits().ResString;
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/28/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2474"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing valid blood units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2475"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of discarded/quarantined blood units
		/// </summary>
		/// <returns></returns>
		public static DataTable GetDiscardOrQuarantineUnits()
		{
			DataTable dt = DAL.DiscardOrQuarantineBloodUnit.GetDiscardOrQuarantineUnits();  
			return dt;
		}

		///<Developers>
		///<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="977"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Object with Details Rule Broken</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="11"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Sets initial broken rules for details
		/// </summary>
		public void SetInitialState()
		{
			RuleBroken("Details", false); 
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1432"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}
		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		/// <summary>
		/// Not implemented
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7083"> 
		///		<ExpectedInput>Valid Blood Unit w/no ABO/Rh Confirmation Tests</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7084"> 
		///		<ExpectedInput>Valid Blood Unit with consistent ABO/Rh Confirmation Tests</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7093"> 
		///		<ExpectedInput>Valid Blood Unit with conflicting ABO Confirmation Test</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7094"> 
		///		<ExpectedInput>Valid Blood Unit with conflicting Rh Confirmation Test</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7118"> 
		///		<ExpectedInput>Invalid Blood Unit</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_27.16
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private bool IsUnitAboRhInconsistent(Guid bloodUnitGuid)
		{
			DataSet ds = DAL.DiscardOrQuarantineBloodUnit.GetAboRhForConsistencyCheck(bloodUnitGuid);
			//
			string bloodUnitMediaAbo = ds.Tables[0].Rows[0][Common.VbecsTables.BloodType.BloodTypeCode].ToString().Trim().ToUpper();
			string bloodUnitMediaRh = ds.Tables[0].Rows[0][Common.VbecsTables.RhFactor.RhFactorText].ToString().Trim().ToUpper();
			//
			// It's possible the unit was quarantined before performing ABO/Rh test, in 
			// which case the bloodUnitTestABORh table will be null and the unit is still 
			// consistent
			//
			// Using current bloodUnitMediaAbo and bloodUnitMediaRh as control, look for all 
			// tests which do not match and return 'true' (inconsistent) if any are found;
			// performing this check for all tests in the table (ds.Tables[1])
			// until either all are validated or an inconsistent test is found
			//
			if(ds.Tables.Count == 2 && ds.Tables[1].Rows.Count > 0)
			{
				string testType = string.Empty;
				string testResults = string.Empty;
				//
				for (int testCtr = 0; testCtr < ds.Tables[1].Rows.Count; testCtr++)
				{
					testType = ds.Tables[1].Rows[testCtr][Common.VbecsTables.BloodTestType.BloodTestName].ToString();
					testResults = ds.Tables[1].Rows[testCtr][Common.VbecsTables.TestResult.TestResultText].ToString().Trim().ToUpper();
					//
					switch (testType)
					{
						case TEST_TYPE_ABO:
						{
							if(!testResults.Equals(string.Empty) && !bloodUnitMediaAbo.Equals(testResults))
							{
								return true;
							}
							break;
						}
						case TEST_TYPE_RH:
						{
							if (bloodUnitMediaRh.ToUpper().Equals("POS") && testResults.ToUpper().Equals("NOT TESTED"))
							{
								//TT2.01.1 - Rh Pos Units -- RH_Testing is OPTIONAL, and "NOT TESTED" is valid.
							}
							else
							{
								if(!testResults.Equals(string.Empty) && !bloodUnitMediaRh.Equals(testResults))
								{
									return true;
								}
							}
							break;
						}
					}
				}
			}
			//
			return false;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/18/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7095"> 
		///		<ExpectedInput>Valid Blood Unit w/no Antigen Typing Tests</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7096"> 
		///		<ExpectedInput>Valid Blood Unit with Blood Unit Antigen info only (no Antigen Typing Tests)</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7097"> 
		///		<ExpectedInput>Valid Blood Unit with consistent Blood Unit Antigen and Antigen Typing Test Data</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7098"> 
		///		<ExpectedInput>Valid Blood Unit with conflicting Blood Unit Antigen and Antigen Typing Test Data</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7116"> 
		///		<ExpectedInput>Valid Blood Unit with conflicting Antigen Typing Tests (no Blood Unit Antigen info)</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7117"> 
		///		<ExpectedInput>Invalid Blood Unit</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_27.16
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private bool IsUnitAntigenTypingInconsistent(Guid bloodUnitGuid)
		{
			DataSet ds = DAL.DiscardOrQuarantineBloodUnit.GetAntigenTypingForConsistencyCheck(bloodUnitGuid);
			//
			if(ds.Tables.Count == 2 && ds.Tables[0].Rows.Count > 0 && ds.Tables[1].Rows.Count > 0)
			{
				DataTable antigenTypes = ds.Tables[0];
				DataTable antigenTests = ds.Tables[1];
				//
				// Using current antigenTypeId as control, look for all the occurrences 
				// of this type and return 'true' (inconsistent) if any do not match;
				// performing this check for all antigen types in the antigenTypes table
				// until either all are validated or an inconsistent test is found
				//
				for (int typeCtr = 0; typeCtr < antigenTypes.Rows.Count; typeCtr++)
				{
					// used to count number of occurrences of current antigenTypeId
					int counter = 0;
					// returned by stored procedure: 'Pos' = 1, 'Neg' = 0
					int currentResult = 0;
					// current id
					int antigenTypeId = (int)antigenTypes.Rows[typeCtr][Common.VbecsTables.BloodUnitAntigen.AntigenTypeId];
					//
					for (int testCtr = 0; testCtr < antigenTests.Rows.Count; testCtr++)
					{
						int antigenTestId = (int)antigenTests.Rows[testCtr][Common.VbecsTables.AntigenTest.AntigenTypeId];
						int antigenTestPositive = (int)antigenTests.Rows[testCtr][Common.VbecsTables.BloodUnitAntigen.Positive];
						//
						if(antigenTypeId == antigenTestId)
						{
							if(counter == 0)
							{
								currentResult = antigenTestPositive;
							}
							else
							{
								if(currentResult != antigenTestPositive)
								{
									return true;
								}
							}
							counter += 1;
						}
					}
				}
			}
			//
			return false;
		}

        /// <summary>
        /// Note: may return no data if Antigen Typing has not been performed
        /// </summary>
        /// <param name="bloodUnitGuid"></param>
        /// <returns></returns>
        public static DataSet GetAntigenTypingForConsistencyCheck(Guid bloodUnitGuid)
        {
            return DAL.DiscardOrQuarantineBloodUnit.GetAntigenTypingForConsistencyCheck(bloodUnitGuid);
        }

		#endregion
	}
}



